<?php
// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}

try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
    // Get and parse JSON input
    $jsonData = getJsonRequest();
    
    // Validate required fields
    if (empty($jsonData['cart']) || !is_array($jsonData['cart'])) {
        throw new Exception("Cart array is required", 400);
    }
    
    $cart = $jsonData["cart"];
    $total = 0;
    
    // Calculate total from real product prices
    foreach ($cart as $cartItem) {
        // Validate cart item structure
        if (empty($cartItem['id']) || empty($cartItem['qty'])) {
            continue; // Skip invalid items
        }
        
        $productId = intval($cartItem['id']);
        $qty = intval($cartItem['qty']);
        
        if ($productId <= 0 || $qty <= 0) {
            continue; // Skip invalid values
        }
        
        // Get product details from database
        $productStmt = $pdo->prepare("
            SELECT p.id, p.itemname, p.price, p.pricepromo, p.status
            FROM mst_product p
            WHERE p.id = :productId AND p.status = 1
        ");
        $productStmt->bindParam(':productId', $productId, PDO::PARAM_INT);
        $productStmt->execute();
        
        if ($productStmt->rowCount() === 0) {
            continue; // Skip if product not found or inactive
        }
        
        $product = $productStmt->fetch(PDO::FETCH_ASSOC);
        
        // Determine final price (use promo price if available and valid)
        $finalPrice = $product['price'];
        if (!empty($product['pricepromo']) && $product['pricepromo'] > 0 && $product['pricepromo'] < $product['price']) {
            $finalPrice = $product['pricepromo'];
        }
        
        // Add to total
        $total += ($qty * $finalPrice);
    }
    
    // Validate that we have items and total > 0
    if ($total <= 0) {
        throw new Exception("Cart is empty or contains invalid items", 400);
    }
    
    // Payment gateway configuration
    $apikey = "xnd_production_H0no0KeHK0mg084eETOYovUftmyRiT07HX4DlM3FerkDAWsmzeUvQEymwgZCQ";
    $accesstoken = base64_encode("$apikey:");
    
    // Generate unique transaction number
    $trxnoadd = mt_rand(1000, 9999);
    $trxno = date("YmdHis") . $trxnoadd;
    $grand = $total;
    
    // Prepare Xendit API request
    $headers = array("Content-Type: application/json", "Authorization: Basic $accesstoken");
    $url = "https://api.xendit.co/v2/invoices";
    
    $reqbody = [
        'external_id' => $trxno,
        'amount' => $grand,
        'description' => 'MATRIX PARABOLA'
    ];
    
    $bodyjson = json_encode($reqbody);
    
    // Make API call to Xendit
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $bodyjson);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    
    $responsecurl = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    // Check for cURL errors
    if ($curlError) {
        throw new Exception("Payment gateway connection error: " . $curlError, 500);
    }
    
    // Check HTTP response code
    if ($httpCode !== 200 && $httpCode !== 201) {
        throw new Exception("Payment gateway error: HTTP " . $httpCode, 500);
    }
    
    // Parse response
    $respdecode = json_decode($responsecurl, true);
    
    if (!$respdecode || !isset($respdecode['invoice_url'])) {
        throw new Exception("Invalid payment gateway response", 500);
    }
    
    $checkouturl = $respdecode['invoice_url'];
    
    // Format the response
    $response = [
        "success" => true,
        "data" => [
            "transaction_id" => $trxno,
            "total_amount" => intval($grand),
            "checkouturl" => $checkouturl
        ]
    ];
    
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("Payment Product API error: " . $e->getMessage());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>