<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}

try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
    // Get and parse JSON input
    $jsonData = getJsonRequest();
    
    // Validate required fields
    if (empty($jsonData['items']) || !is_array($jsonData['items'])) {
        throw new Exception("Items array is required", 400);
    }
    
    $cart = $jsonData["items"];
    $items = array();
    
    // Get base URL from config for product images
    $baseUrl = getenv('BASE_URL') ?: 'https://api.ezlink.dev/matrixweb/assets';
    
    foreach ($cart as $cartItem) {
        // Validate cart item structure
        if (empty($cartItem['id']) || empty($cartItem['qty'])) {
            continue; // Skip invalid items
        }
        
        $productId = intval($cartItem['id']);
        $qty = intval($cartItem['qty']);
        
        if ($productId <= 0 || $qty <= 0) {
            continue; // Skip invalid values
        }
        
        // Get product details from database
        $productStmt = $pdo->prepare("
            SELECT p.id, p.itemname, p.price, p.pricepromo, p.stock, p.status
            FROM mst_product p
            WHERE p.id = :productId AND p.status = 1
        ");
        $productStmt->bindParam(':productId', $productId, PDO::PARAM_INT);
        $productStmt->execute();
        
        if ($productStmt->rowCount() === 0) {
            continue; // Skip if product not found or inactive
        }
        
        $product = $productStmt->fetch(PDO::FETCH_ASSOC);
        
        // Get product photo
        $photoStmt = $pdo->prepare("
            SELECT photo 
            FROM mst_productphoto 
            WHERE productid = :productId 
            ORDER BY seqno ASC 
            LIMIT 1
        ");
        $photoStmt->bindParam(':productId', $productId, PDO::PARAM_INT);
        $photoStmt->execute();
        $photo = $photoStmt->fetch(PDO::FETCH_ASSOC);
        
        // Set image URL
        if ($photo && !empty($photo['photo'])) {
            $imageUrl = $baseUrl . '/product/' . $photo['photo'];
        } else {
            // Default image if product has no photos
            $imageUrl = $baseUrl . '/home/default-product.jpg';
        }
        
        // Determine final price (use promo price if available and valid)
        $finalPrice = $product['price'];
        if (!empty($product['pricepromo']) && $product['pricepromo'] > 0 && $product['pricepromo'] < $product['price']) {
            $finalPrice = $product['pricepromo'];
        }
        
        // Create cart item object
        $cartItemObj = array(
            'id' => $productId,
            'qty' => $qty,
            'productname' => $product['itemname'],
            'price' => intval($finalPrice),
            'imageurl' => $imageUrl,
            'stock' => intval($product['stock'])
        );
        
        array_push($items, $cartItemObj);
    }
    
    // Format the response
    $response = [
        "success" => true,
        "data" => [
            "items" => $items
        ]
    ];
    
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("Cart API error: " . $e->getMessage());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>