<?php
// Strict error reporting in development, disable in production
ini_set('display_errors', 0);
error_reporting(E_ALL);

// Include database connection function
require_once __DIR__ . "/auth.php";
require_once __DIR__ . "/connection.php";

// Set CORS headers
setCorsHeaders();

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}

header('Content-Type: application/json; charset=UTF-8');

// Initialize response
$response = [
    "success" => false,
    "message" => "",
    "data" => null
];

try {
    // Get PDO connection
    $pdo = connectDB();
    
     // Validate API key
     validateApiKey();
    
     // Get and parse JSON input
     $jsonData = getJsonRequest();

    // Validate required fields
    if (empty($jsonData['userid']) || empty($jsonData['userpwd'])) {
        throw new Exception("Missing required fields", 400);
    }
    
    // Sanitize input
    $userid = trim($jsonData['userid']);
    $password = $jsonData['userpwd'];
    
    // Get user from database
    $stmt = $pdo->prepare("SELECT id, username, userpwd, account_locked, failed_attempts, last_attempt 
                           FROM mst_user 
                           WHERE userid = :userid");
    $stmt->bindParam(':userid', $userid, PDO::PARAM_STR);
    $stmt->execute();
    
    if ($stmt->rowCount() === 0) {
        // Use consistent timing to prevent username enumeration
        password_verify($password, password_hash('dummy', PASSWORD_DEFAULT));
        throw new Exception("Invalid credentials", 401);
    }
    
    $user = $stmt->fetch();
    
    // Check account lockout
    if (!empty($user['account_locked']) && $user['account_locked'] == 1) {
        // Check if lockout period has expired (30 minutes)
        $lockoutTime = strtotime($user['last_attempt']) + (30 * 60);
        if (time() < $lockoutTime) {
            throw new Exception("Account temporarily locked. Please try again later.", 403);
        } else {
            // Reset lockout if time has expired
            $resetStmt = $pdo->prepare("UPDATE mst_user SET account_locked = 0, failed_attempts = 0 WHERE id = :id");
            $resetStmt->bindParam(':id', $user['id'], PDO::PARAM_INT);
            $resetStmt->execute();
        }
    }
    
    // Verify password
    if (!password_verify($password, $user['userpwd'])) {
        // Increment failed attempts
        $maxAttempts = $_ENV['MAX_LOGIN_ATTEMPTS'] ?? 5;
        $failedAttempts = ($user['failed_attempts'] ?? 0) + 1;
        $accountLocked = ($failedAttempts >= $maxAttempts) ? 1 : 0;
        
        $failStmt = $pdo->prepare("UPDATE mst_user SET failed_attempts = :attempts, account_locked = :locked, 
                                  last_attempt = NOW() WHERE id = :id");
        $failStmt->bindParam(':attempts', $failedAttempts, PDO::PARAM_INT);
        $failStmt->bindParam(':locked', $accountLocked, PDO::PARAM_INT);
        $failStmt->bindParam(':id', $user['id'], PDO::PARAM_INT);
        $failStmt->execute();
        
        throw new Exception("Invalid credentials", 401);
    }
    
    // Reset failed attempts on successful login
    $resetStmt = $pdo->prepare("UPDATE mst_user SET failed_attempts = 0, account_locked = 0 WHERE id = :id");
    $resetStmt->bindParam(':id', $user['id'], PDO::PARAM_INT);
    $resetStmt->execute();
    
    // Generate token
    $token = bin2hex(random_bytes(32));
    $expiryTime = time() + ($_ENV['TOKEN_LIFETIME_HOURS'] ?? 24) * 60 * 60;
    
    // Store token
    $tokenStmt = $pdo->prepare("UPDATE mst_user SET sessionid = :token, token_expiry = FROM_UNIXTIME(:expiry) WHERE id = :id");
    $tokenStmt->bindParam(':token', $token, PDO::PARAM_STR);
    $tokenStmt->bindParam(':expiry', $expiryTime, PDO::PARAM_INT);
    $tokenStmt->bindParam(':id', $user['id'], PDO::PARAM_INT);
    $tokenStmt->execute();
    
    // Success response
    $response["success"] = true;
    $response["message"] = "Login successful";
    $response["data"] = [
        "userid" => intval($user['id']),
        "username" => $user['username'],
        "token" => $token,
        "expires" => $expiryTime
    ];
    
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : intval($errorCode);
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("Login API error: " . $e->getMessage());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}

?>