<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}

try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
    // Query to get userguide data
    $query = "
        SELECT id, docname, note, filedoc
        FROM mst_userguide
        ORDER BY id DESC
    ";
    
    // Execute query
    $stmt = $pdo->prepare($query);
    $stmt->execute();
    $userguideList = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get base URL from config for file downloads
    $baseUrl = getenv('BASE_URL') ?: 'https://api.ezlink.dev/matrixweb/assets';
    $baseUrl = rtrim($baseUrl, '/') . '/userguide/';
    
    // Format the response
    $userguide = array();
    foreach ($userguideList as $item) {
        $userguideObj = array(
            'id' => intval($item['id']),
            'title' => $item['docname'] ?? '',
            'briefdesc' => $item['note'] ?? '',
            'pdffile' => !empty($item['filedoc']) ? $baseUrl . $item['filedoc'] : null
        );
        
        array_push($userguide, $userguideObj);
    }
    
    // Format the response
    $response = [
        "success" => true,
        "data" => [
            "userguide" => $userguide
        ]
    ];
    
    // Send successful response
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("User Guide API error: " . $e->getMessage() . "\n" . $e->getTraceAsString());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>