<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}



try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
    // Validate authentication token
    $authenticatedUser = validateAuthToken($pdo);
    
    // Validate required parameter
    if (!isset($_GET['categoryid']) || empty($_GET['categoryid'])) {
        throw new Exception("Category ID is required", 400);
    }
    
    // Get category ID
    $categoryId = intval($_GET['categoryid']);
    
    // Check if category exists
    $checkCategoryStmt = $pdo->prepare("SELECT id FROM mst_category WHERE id = :categoryid");
    $checkCategoryStmt->bindParam(':categoryid', $categoryId, PDO::PARAM_INT);
    $checkCategoryStmt->execute();
    
    if ($checkCategoryStmt->rowCount() === 0) {
        throw new Exception("Category not found", 404);
    }
    
    // Query subcategories for the given category ID
    $query = "
        SELECT id, categoryid, subcategory 
        FROM mst_subcategory 
        WHERE categoryid = :categoryid 
        ORDER BY subcategory ASC
    ";
    
    $stmt = $pdo->prepare($query);
    $stmt->bindParam(':categoryid', $categoryId, PDO::PARAM_INT);
    $stmt->execute();
    $subcategories = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Format the response
    $response = [
        "success" => true,
        "data" => $subcategories
    ];
    
    // Send successful response
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("List Subcategories API error: " . $e->getMessage() . "\n" . $e->getTraceAsString());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>