<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}

try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
    // Query to get software data
    $query = "
        SELECT id, docname, note, filedoc
        FROM mst_software
        ORDER BY id DESC
    ";
    
    // Execute query
    $stmt = $pdo->prepare($query);
    $stmt->execute();
    $softwareList = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get base URL from config for file downloads
    $baseUrl = getenv('BASE_URL') ?: 'https://api.ezlink.dev/matrixweb/assets';
    $baseUrl = rtrim($baseUrl, '/') . '/software/';
    
    // Format the response
    $software = array();
    foreach ($softwareList as $item) {
        $softwareObj = array(
            'id' => intval($item['id']),
            'title' => $item['docname'] ?? '',
            'briefdesc' => $item['note'] ?? '',
            'fileupdate' => !empty($item['filedoc']) ? $baseUrl . $item['filedoc'] : null
        );
        
        array_push($software, $softwareObj);
    }
    
    // Format the response
    $response = [
        "success" => true,
        "data" => [
            "software" => $software
        ]
    ];
    
    // Send successful response
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("Software Update API error: " . $e->getMessage() . "\n" . $e->getTraceAsString());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>