<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}

try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
    // Validate required parameter - accept productid, sku, or slug
    $productId = null;
    $whereClause = "";
    $paramValue = null;
    
    if (!empty($_GET['productid'])) {
        $productId = intval($_GET['productid']);
        $whereClause = "p.id = :paramValue";
        $paramValue = $productId;
    } elseif (!empty($_GET['sku'])) {
        $whereClause = "p.sku = :paramValue";
        $paramValue = trim($_GET['sku']);
    } elseif (!empty($_GET['slug'])) {
        $whereClause = "p.slug = :paramValue";
        $paramValue = trim($_GET['slug']);
    } else {
        throw new Exception("Product identifier is required (productid, sku, or slug)", 400);
    }
    
    // Get product details
    $productStmt = $pdo->prepare("
        SELECT p.id, p.sku, p.barcode, p.itemname, p.briefdesc, p.price, p.pricepromo, p.sku, p.slug, 
              p.specification, p.stock, p.status, b.brand, p.categoryid
        FROM mst_product p
        LEFT JOIN mst_brand b ON p.brandid = b.id
        WHERE {$whereClause}
    ");
    $productStmt->bindParam(':paramValue', $paramValue);
    $productStmt->execute();
    
    if ($productStmt->rowCount() === 0) {
        throw new Exception("Product not found", 404);
    }
    
    $product = $productStmt->fetch(PDO::FETCH_ASSOC);
    $productId = $product['id']; // Set productId for further queries
    $slug = $product['slug']; // Set productId for further queries
    $sku = $product['sku']; // Set productId for further queries
    
    // Get product photos
    $photoStmt = $pdo->prepare("
        SELECT id, photo
        FROM mst_productphoto
        WHERE productid = :productId
        ORDER BY seqno ASC
    ");
    $photoStmt->bindParam(':productId', $productId, PDO::PARAM_INT);
    $photoStmt->execute();
    
    $photos = $photoStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get base URL from config for photo images
    $baseUrl = getenv('BASE_URL') ?: 'https://api.ezlink.dev/matrixweb/assets';
    
    // Format photos with full URLs
    $photoData = [];
    if (!empty($photos)) {
        foreach ($photos as $photo) {
            $photoData[] = [
                'id' => intval($photo['id']),
                'imageurl' => $baseUrl . '/product/' . $photo['photo']
            ];
        }
    } else {
        // If no photos found, add a placeholder
        $photoData[] = [
            'id' => 0,
            'imageurl' => $baseUrl . '/home/default-product.jpg'
        ];
    }
    
    // Get rating and review count for product
    $ratingStmt = $pdo->prepare("
        SELECT AVG(rating) as avg_rating, COUNT(*) as total_reviews 
        FROM mst_productreview 
        WHERE productid = :productId
    ");
    $ratingStmt->bindParam(':productId', $productId, PDO::PARAM_INT);
    $ratingStmt->execute();
    $ratingResult = $ratingStmt->fetch(PDO::FETCH_ASSOC);
    $avgRating = $ratingResult['avg_rating'] ? round($ratingResult['avg_rating'], 1) : 0;
    $totalReviews = intval($ratingResult['total_reviews']);
    
    // Format price for display
    if ($product['pricepromo'] > 0 && $product['pricepromo'] < $product['price']) {
        $displayPrice = "Rp. " . number_format($product['pricepromo'], 0);
        $normalPrice = "Rp. " . number_format($product['price'], 0);
    } else {
        $displayPrice = "Rp. " . number_format($product['price'], 0);
        $normalPrice = "";
    }
    
    // Get similar products (products from the same category)
    $similarProducts = [];
    
    // First, check if there are explicitly defined similar products
    $similarStmt = $pdo->prepare("
        SELECT p.id, p.itemname, p.briefdesc, p.price, p.pricepromo, p.slug
        FROM mst_productsimilar ps
        JOIN mst_product p ON ps.productsimilarid = p.id
        WHERE ps.productid = :productId
        ORDER BY p.itemname ASC
    ");
    $similarStmt->bindParam(':productId', $productId, PDO::PARAM_INT);
    $similarStmt->execute();
    $explicitSimilarProducts = $similarStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // If no explicit similar products, get products from the same category
    if (empty($explicitSimilarProducts) && !empty($product['categoryid'])) {
        $categorySimilarStmt = $pdo->prepare("
            SELECT id, itemname, briefdesc, price, pricepromo, slug
            FROM mst_product
            WHERE categoryid = :categoryId 
            AND id != :productId
            AND status = 1
            ORDER BY itemname ASC
            LIMIT 5
        ");
        $categorySimilarStmt->bindParam(':categoryId', $product['categoryid'], PDO::PARAM_INT);
        $categorySimilarStmt->bindParam(':productId', $productId, PDO::PARAM_INT);
        $categorySimilarStmt->execute();
        $explicitSimilarProducts = $categorySimilarStmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Format similar products
    foreach ($explicitSimilarProducts as $similarProd) {
        // Get first photo for this similar product
        $similarPhotoStmt = $pdo->prepare("
            SELECT photo 
            FROM mst_productphoto 
            WHERE productid = :productId 
            ORDER BY seqno ASC 
            LIMIT 1
        ");
        $similarPhotoStmt->bindParam(':productId', $similarProd['id'], PDO::PARAM_INT);
        $similarPhotoStmt->execute();
        $similarPhoto = $similarPhotoStmt->fetch(PDO::FETCH_ASSOC);
        
        // Set image URL
        if ($similarPhoto && !empty($similarPhoto['photo'])) {
            $imageUrl = $baseUrl . '/product/' . $similarPhoto['photo'];
        } else {
            $imageUrl = $baseUrl . '/home/default-product.jpg';
        }
        
        // Format prices
        if ($similarProd['pricepromo'] > 0 && $similarProd['pricepromo'] < $similarProd['price']) {
            $similarPrice = "Rp. " . number_format($similarProd['pricepromo'], 0);
            $similarNormalPrice = "Rp. " . number_format($similarProd['price'], 0);
        } else {
            $similarPrice = "Rp. " . number_format($similarProd['price'], 0);
            $similarNormalPrice = "";
        }
        
        // Get average rating for similar product
        $similarRatingQuery = "
            SELECT AVG(rating) as avg_rating 
            FROM mst_productreview 
            WHERE productid = :productId
        ";
        $similarRatingStmt = $pdo->prepare($similarRatingQuery);
        $similarRatingStmt->bindParam(':productId', $similarProd['id'], PDO::PARAM_INT);
        $similarRatingStmt->execute();
        $similarRatingResult = $similarRatingStmt->fetch(PDO::FETCH_ASSOC);
        $similarAvgRating = $similarRatingResult['avg_rating'] ? round($similarRatingResult['avg_rating'], 1) : 0;
        
        // Add to similar products array
        $similarProducts[] = [
            'productid' => intval($similarProd['id']),
            'productname' => $similarProd['itemname'],
            'briefdesc' => $similarProd['briefdesc'] ?? '',
            'price' => $similarPrice,
            'pricenormal' => $similarNormalPrice,
            'imageurl' => $imageUrl,
            'slug' => $similarProd['slug'],
            'rating' => $similarAvgRating
        ];
    }
    
    // Format the response
    $response = [
        "success" => true,
        "data" => [
            "photo" => $photoData,
            "productname" => $product['itemname'],
            "productid" => $productId,
            "sku" => $sku,
            "slug" => $slug,
            "briefdesc" => $product['briefdesc'] ?? '',
            "price" => $displayPrice,
            "pricenormal" => $normalPrice,
            "specification" => $product['specification'] ?? '',
            "stock" => intval($product['stock']),
            "rating" => $avgRating,
            "totalreview" => $totalReviews,
            "similarproducts" => $similarProducts
        ]
    ];
    
    // Send successful response
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("View Product API error: " . $e->getMessage());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>