<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}



try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
    // Validate required parameter
    if (!isset($_GET['productid']) || empty($_GET['productid'])) {
        throw new Exception("Product ID is required", 400);
    }
    
    $productId = intval($_GET['productid']);
    
    // Pagination parameters
    $page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
    $limit = isset($_GET['limit']) ? max(1, min(100, intval($_GET['limit']))) : 10;
    $offset = ($page - 1) * $limit;
    
    // Verify product exists
    $productCheckStmt = $pdo->prepare("SELECT id FROM mst_product WHERE id = :productid");
    $productCheckStmt->bindParam(':productid', $productId, PDO::PARAM_INT);
    $productCheckStmt->execute();
    
    if ($productCheckStmt->rowCount() === 0) {
        throw new Exception("Product not found", 200);
    }
    
    // Count total reviews for this product
    $countQuery = "
        SELECT COUNT(*) as total 
        FROM mst_productreview 
        WHERE productid = :productid
    ";
    $countStmt = $pdo->prepare($countQuery);
    $countStmt->bindParam(':productid', $productId, PDO::PARAM_INT);
    $countStmt->execute();
    $totalData = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Get reviews with pagination
    $reviewQuery = "
        SELECT id, trxdate, username, review, rating
        FROM mst_productreview
        WHERE productid = :productid
        ORDER BY trxdate DESC
        LIMIT :limit OFFSET :offset
    ";
    
    $reviewStmt = $pdo->prepare($reviewQuery);
    $reviewStmt->bindParam(':productid', $productId, PDO::PARAM_INT);
    $reviewStmt->bindParam(':limit', $limit, PDO::PARAM_INT);
    $reviewStmt->bindParam(':offset', $offset, PDO::PARAM_INT);
    $reviewStmt->execute();
    $reviews = $reviewStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Format reviews
    $formattedReviews = array();
    foreach ($reviews as $review) {
        $reviewObj = array(
            'id' => intval($review['id']),
            'trxdate' => !empty($review['trxdate']) ? date('Y-m-d', strtotime($review['trxdate'])) : null,
            'username' => $review['username'] ?? '',
            'review' => $review['review'] ?? '',
            'rating' => floatval($review['rating'])
        );
        
        $formattedReviews[] = $reviewObj;
    }
    
    // Calculate average rating
    $ratingStatsQuery = "
        SELECT 
            AVG(rating) as avg_rating,
            COUNT(*) as total_reviews
        FROM mst_productreview 
        WHERE productid = :productid
    ";
    $ratingStatsStmt = $pdo->prepare($ratingStatsQuery);
    $ratingStatsStmt->bindParam(':productid', $productId, PDO::PARAM_INT);
    $ratingStatsStmt->execute();
    $ratingStats = $ratingStatsStmt->fetch(PDO::FETCH_ASSOC);
    
    // Format rating statistics
    $avgRating = $ratingStats['avg_rating'] ? round($ratingStats['avg_rating'], 1) : 0;
    $totalReviews = intval($ratingStats['total_reviews']);
    
    // Format the response
    $response = [
        "success" => true,
        "data" => [
            "reviews" => $formattedReviews,
            "statistics" => [
                "average_rating" => $avgRating,
                "total_reviews" => $totalReviews
            ],
            "pagination" => [
                "total_data" => intval($totalData),
                "current_page" => $page,
                "limit" => $limit,
                "total_pages" => ceil($totalData / $limit)
            ]
        ]
    ];
    
    // Send successful response
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("Product Review API error: " . $e->getMessage());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>