<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}

try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
    // Validate required parameter
    if (!isset($_GET['golonganid']) || empty($_GET['golonganid'])) {
        throw new Exception("Golongan ID is required", 400);
    }
    
    // Get golongan ID
    $golonganId = intval($_GET['golonganid']);
    
    // Initialize the product array
    $product = array();
    
    // Get categories that have products associated with this golongan
    $categoryQuery = "
        SELECT DISTINCT c.id, c.category 
        FROM mst_category c
        INNER JOIN mst_product p ON c.id = p.categoryid
        WHERE p.golonganid = :golonganid AND p.status = 1
        ORDER BY c.category ASC
    ";
    $categoryStmt = $pdo->prepare($categoryQuery);
    $categoryStmt->bindParam(':golonganid', $golonganId, PDO::PARAM_INT);
    $categoryStmt->execute();
    $categories = $categoryStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Prepare base URL for images
    $baseUrl = getenv('BASE_URL') ?: 'https://api.ezlink.dev/matrixweb/assets';
    
    // Loop through each category
    foreach ($categories as $category) {
        // Create category object with empty detail array
        $categoryObj = array(
            'id' => intval($category['id']),
            'category' => $category['category'],
            'detail' => array()
        );
        
        // Get products for this category and golongan
        $productQuery = "
            SELECT p.id, p.itemname, p.briefdesc, p.price, p.pricepromo, p.slug
            FROM mst_product p
            WHERE p.categoryid = :categoryid AND p.golonganid = :golonganid AND p.status = 1
            ORDER BY p.itemname ASC
        ";
        $productStmt = $pdo->prepare($productQuery);
        $productStmt->bindParam(':categoryid', $category['id'], PDO::PARAM_INT);
        $productStmt->bindParam(':golonganid', $golonganId, PDO::PARAM_INT);
        $productStmt->execute();
        $products = $productStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Process products for this category
        foreach ($products as $prod) {
            // Get the first photo for this product
            $photoQuery = "
                SELECT photo 
                FROM mst_productphoto 
                WHERE productid = :productid 
                ORDER BY seqno ASC 
                LIMIT 1
            ";
            $photoStmt = $pdo->prepare($photoQuery);
            $photoStmt->bindParam(':productid', $prod['id'], PDO::PARAM_INT);
            $photoStmt->execute();
            $photo = $photoStmt->fetch(PDO::FETCH_ASSOC);
            
            // Set image URL
            if ($photo && !empty($photo['photo'])) {
                $imageUrl = $baseUrl . '/product/' . $photo['photo'];
            } else {
                // Default image if product has no photos
                $imageUrl = $baseUrl . '/home/default-product.jpg';
            }
            
            // Get average rating for product
            $ratingQuery = "
                SELECT AVG(rating) as avg_rating 
                FROM mst_productreview 
                WHERE productid = :productid
            ";
            $ratingStmt = $pdo->prepare($ratingQuery);
            $ratingStmt->bindParam(':productid', $prod['id'], PDO::PARAM_INT);
            $ratingStmt->execute();
            $ratingResult = $ratingStmt->fetch(PDO::FETCH_ASSOC);
            $avgRating = $ratingResult['avg_rating'] ? round($ratingResult['avg_rating'], 1) : 0;
            
            // Create product object
            if (isset($prod['pricepromo']) && $prod['pricepromo'] > 0 && $prod['pricepromo'] < $prod['price'])
            {
                $price = "Rp. " . number_format($prod['pricepromo'], 0);
                $pricenormal = "Rp. " . number_format($prod['price'], 0);
            }
            else
            {
                $price = "Rp. " . number_format($prod['price'], 0);
                $pricenormal = "";
            }
            $productObj = array(
                'productid' => intval($prod['id']),
                'productname' => $prod['itemname'],
                'briefdesc' => $prod['briefdesc'],
                'price' => $price,
                'pricenormal' => $pricenormal,
                'imageurl' => $imageUrl,
                'slug' => $prod['slug'],
                'rating' => $avgRating
            );
            
            // Add to category's detail array
            $categoryObj['detail'][] = $productObj;
        }
        
        // Add category to product array
        $product[] = $categoryObj;
    }
    
    // Format the response exactly as in the original
    $response = [
        "success" => true,
        "data" => [
            "product" => $product
        ]
    ];
    
    // Send successful response
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>