<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}

try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
    // Validate authentication token
    $authenticatedUser = validateAuthToken($pdo);
    
    // Validate required parameter
    if (!isset($_GET['id']) || empty($_GET['id'])) {
        throw new Exception("Paket ID is required", 400);
    }
    
    // Get paket ID
    $paketId = intval($_GET['id']);
    
    // Query paket data with category name
    $paketStmt = $pdo->prepare("
        SELECT p.*, c.category
        FROM mst_paketnex p
        LEFT JOIN mst_nexcategory c ON p.categoryid = c.id
        WHERE p.id = :id 
    ");
    $paketStmt->bindParam(':id', $paketId, PDO::PARAM_INT);
    $paketStmt->execute();
    
    // Check if paket exists
    if ($paketStmt->rowCount() === 0) {
        throw new Exception("Paket not found", 404);
    }
    
    // Get paket data with category name
    $paketData = $paketStmt->fetch(PDO::FETCH_ASSOC);
    
    // Format dates
    if (!empty($paketData['startdate'])) {
        $paketData['startdate'] = date('Y-m-d', strtotime($paketData['startdate']));
    }
    if (!empty($paketData['enddate'])) {
        $paketData['enddate'] = date('Y-m-d', strtotime($paketData['enddate']));
    }
    
    // Get base URL from config for banner images
    $baseUrl = getenv('BASE_URL') ?: 'https://api.ezlink.dev/matrixweb/assets';
    
    // Set full banner URL if banner exists
    if (!empty($paketData['bannerfile'])) {
        $paketData['banner_url'] = $baseUrl . '/paketnex/' . $paketData['bannerfile'];
    } else {
        $paketData['banner_url'] = null;
    }
    
    // Format the response
    $response = [
        "success" => true,
        "message" => "Paket retrieved successfully",
        "data" => $paketData
    ];
    
    // Send successful response
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("View Paket Nex API error: " . $e->getMessage());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>