<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}



try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
    // Validate authentication token
    $authenticatedUser = validateAuthToken($pdo);
    
    // Pagination parameters
    $isPaging = isset($_GET['is_paging']) ? filter_var($_GET['is_paging'], FILTER_VALIDATE_BOOLEAN) : true;
    $page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
    $pageSize = isset($_GET['page_size']) ? max(1, min(100, intval($_GET['page_size']))) : 10;
    $offset = ($page - 1) * $pageSize;
    
    // Search parameter
    $search = isset($_GET['search']) ? trim($_GET['search']) : '';
    
    // Category filter
    $categoryFilter = isset($_GET['categoryid']) && !empty($_GET['categoryid']) ? intval($_GET['categoryid']) : null;
       
    // Base query with JOIN to get category name
    $baseWhere = "WHERE 1=1";
    
    if ($categoryFilter !== null) {
        $baseWhere .= " AND p.categoryid = $categoryFilter";
    }
        
    $query = "
        SELECT p.id, p.kodepaket, p.namapaket, p.categoryid, c.category, 
               p.masaberlaku, p.price, p.pricepromo, p.startdate, p.enddate, p.bannerfile
        FROM mst_paketnex p
        LEFT JOIN mst_nexcategory c ON p.categoryid = c.id
        $baseWhere
    ";
    
    $countQuery = "
        SELECT COUNT(*) as total
        FROM mst_paketnex p
        LEFT JOIN mst_nexcategory c ON p.categoryid = c.id
        $baseWhere
    ";
    
    // Add search condition if provided
    if (!empty($search)) {
        $searchTerm = "%$search%";
        $searchCondition = " AND (p.kodepaket LIKE '$searchTerm' OR p.namapaket LIKE '$searchTerm')";
        $query .= $searchCondition;
        $countQuery .= $searchCondition;
    }
    
    // Sorting parameters
    $allowedSortFields = ['id', 'kodepaket', 'namapaket', 'category', 'price', 'pricepromo', 'startdate', 'enddate'];
    $sortBy = isset($_GET['sort_by']) && in_array($_GET['sort_by'], $allowedSortFields) ? $_GET['sort_by'] : 'id';
    $sortKey = isset($_GET['sort_key']) && strtoupper($_GET['sort_key']) === 'ASC' ? 'ASC' : 'DESC';
    
    // Execute count query
    $countStmt = $pdo->prepare($countQuery);
    $countStmt->execute();
    $totalCount = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Add sorting
    $query .= " ORDER BY " . ($sortBy === 'category' ? 'c.category' : "p.$sortBy") . " $sortKey";
    
    // Add pagination if enabled
    if ($isPaging) {
        $query .= " LIMIT $pageSize OFFSET $offset";
    }
    
    // Execute main query
    $stmt = $pdo->prepare($query);
    $stmt->execute();
    $paketnexes = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Calculate pagination details
    $totalPages = ceil($totalCount / $pageSize);
    
    // Format the response
    $response = [
        "success" => true,
        "data" => [
            "list" => $paketnexes,
            "sort" => [
                "sort_by" => $sortBy,
                "sort_key" => $sortKey
            ]
        ]
    ];
    
    // Add pagination details if paging is enabled
    if ($isPaging) {
        $response["data"]["pagination"] = [
            "total" => $totalCount,
            "page_size" => $pageSize,
            "current_page" => $page,
            "last_page" => $totalPages,
            "from" => ($totalCount > 0) ? $offset + 1 : 0,
            "to" => min($offset + $pageSize, $totalCount)
        ];
    } else {
        $response["data"]["pagination"] = [
            "total" => $totalCount
        ];
    }
    
    // Send successful response
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("List Paket Nex API error: " . $e->getMessage() . "\n" . $e->getTraceAsString());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>