<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}

try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
    // Validate required parameter
    if (!isset($_GET['categoryid']) || empty($_GET['categoryid'])) {
        throw new Exception("Category ID is required", 400);
    }
    
    $categoryId = intval($_GET['categoryid']);
    
    // Get base URL from config for banner images
    $baseUrl = getenv('BASE_URL') ?: 'https://api.ezlink.dev/matrixweb/assets';
    
    // Get current date for filtering active packages
    $currentDate = date('Y-m-d');
    
    // Get paket nex from database filtered by categoryid
    $productnex = array();
    
    $paketQuery = "
        SELECT p.id, p.namapaket, p.bannerfile, nc.category as categoryname
        FROM mst_paketnex p
        LEFT JOIN mst_nexcategory nc ON p.categoryid = nc.id
        WHERE p.categoryid = :categoryid
        AND (p.startdate <= :currentDate OR p.startdate IS NULL) 
        AND (p.enddate >= :currentDate OR p.enddate IS NULL)
        ORDER BY p.namapaket ASC
    ";
    
    $paketStmt = $pdo->prepare($paketQuery);
    $paketStmt->bindParam(':categoryid', $categoryId, PDO::PARAM_INT);
    $paketStmt->bindParam(':currentDate', $currentDate, PDO::PARAM_STR);
    $paketStmt->execute();
    $paketList = $paketStmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach ($paketList as $paket) {
        // Get paket details (pricing options)
        $detailQuery = "
            SELECT id, kodepaket, masaberlaku, price
            FROM mst_paketnexdet
            WHERE paketid = :paketid
            ORDER BY price ASC
        ";
        
        $detailStmt = $pdo->prepare($detailQuery);
        $detailStmt->bindParam(':paketid', $paket['id'], PDO::PARAM_INT);
        $detailStmt->execute();
        $paketDetails = $detailStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Find minimum price for display
        $minPrice = 0;
        if (!empty($paketDetails)) {
            $minPrice = min(array_column($paketDetails, 'price'));
        }
        
        // Set banner image URL
        $imageUrl = $baseUrl . '/paketnex/default.jpg'; // Default image
        if (!empty($paket['bannerfile'])) {
            $imageUrl = $baseUrl . '/paketnex/' . $paket['bannerfile'];
        }
        
        // Format paket details
        $formattedDetails = array();
        foreach ($paketDetails as $detail) {
            $detailObj = array(
                'kodepaket' => $detail['kodepaket'],
                'masaberlaku' => $detail['masaberlaku'],
                'price' => "Rp. " . number_format($detail['price'], 0)
            );
            $formattedDetails[] = $detailObj;
        }
        
        // Create paket object
        $paketObj = array(
            'id' => intval($paket['id']),
            'paketname' => $paket['namapaket'],
            'paketdesc' => $paket['categoryname'] ?? 'Paket berlangganan',
            'price' => $minPrice > 0 ? "Mulai Rp. " . number_format($minPrice, 0) : "Hubungi Kami",
            'imageurl' => $imageUrl,
            'detail' => $formattedDetails
        );
        
        $productnex[] = $paketObj;
    }
    
    // Format the response
    $response = [
        "success" => true,
        "data" => [
            "productnex" => $productnex
        ]
    ];
    
    // Send successful response
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("Paket API error: " . $e->getMessage());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>