<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}



try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
    // Validate authentication token
    $authenticatedUser = validateAuthToken($pdo);
    
    // Validate required parameter
    if (!isset($_GET['id']) || empty($_GET['id'])) {
        throw new Exception("Category ID is required", 400);
    }
    
    // Get category ID
    $categoryId = intval($_GET['id']);
    
    // Query category data
    $categoryStmt = $pdo->prepare("
        SELECT id, category
        FROM mst_nexcategory
        WHERE id = :id 
    ");
    $categoryStmt->bindParam(':id', $categoryId, PDO::PARAM_INT);
    $categoryStmt->execute();
    
    // Check if category exists
    if ($categoryStmt->rowCount() === 0) {
        throw new Exception("Category not found", 404);
    }
    
    $categoryData = $categoryStmt->fetch(PDO::FETCH_ASSOC);
    
    // Format the response
    $response = [
        "success" => true,
        "message" => "Category retrieved successfully",
        "data" => $categoryData
    ];
    
    // Send successful response
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("View Nexcategory API error: " . $e->getMessage());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>