<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}



try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
    // Validate authentication token
    $authenticatedUser = validateAuthToken($pdo);
    
    // Pagination parameters
    $isPaging = isset($_GET['ispaging']) ? filter_var($_GET['ispaging'], FILTER_VALIDATE_BOOLEAN) : true;
    $page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
    $pageSize = isset($_GET['page_size']) ? max(1, min(100, intval($_GET['page_size']))) : 10;
    $offset = ($page - 1) * $pageSize;
    
    // Search parameter
    $search = isset($_GET['search']) ? trim($_GET['search']) : '';
       
    // Base query
    $baseWhere = "WHERE 1=1";
        
    $query = "
        SELECT id, category
        FROM mst_nexcategory
        $baseWhere
    ";
    $countQuery = "
        SELECT COUNT(*) as total
        FROM mst_nexcategory
        $baseWhere
    ";
    
    // Add search condition if provided
    $searchParams = [];
    if (!empty($search)) {
        $searchTerm = "%$search%";
        $query .= " AND (category LIKE '$searchTerm')";
        $countQuery .= " AND (category LIKE '$searchTerm')";
    }
    
    // Sorting parameters
    $allowedSortFields = ['id', 'category'];
    $sortBy = isset($_GET['sort_by']) && in_array($_GET['sort_by'], $allowedSortFields) ? $_GET['sort_by'] : 'id';
    $sortKey = isset($_GET['sort_key']) && strtoupper($_GET['sort_key']) === 'ASC' ? 'ASC' : 'DESC';
    
    // Execute count query
    $countStmt = $pdo->prepare($countQuery);
    $countStmt->execute();
    $totalCount = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Add sorting and pagination
    if ($isPaging) {
        $query .= " ORDER BY $sortBy $sortKey LIMIT $pageSize OFFSET $offset";
    } else {
        $query .= " ORDER BY $sortBy $sortKey";
    }
    
    // Execute main query
    $stmt = $pdo->prepare($query);
    $stmt->execute();
    $categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Calculate pagination details
    $totalPages = ceil($totalCount / $pageSize);
    
    // Format the response
    $response = [
        "success" => true,
        "data" => [
            "list" => $categories
        ]
    ];
    
    // Add pagination data if paging is enabled
    if ($isPaging) {
        $response['data']['pagination'] = [
            "total" => $totalCount,
            "page_size" => $pageSize,
            "current_page" => $page,
            "last_page" => $totalPages,
            "from" => ($totalCount > 0) ? $offset + 1 : 0,
            "to" => min($offset + $pageSize, $totalCount)
        ];
    }
    
    // Add sorting data
    $response['data']['sort'] = [
        "sort_by" => $sortBy,
        "sort_key" => $sortKey
    ];
    
    // Send successful response
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("List Nexcategory API error: " . $e->getMessage() . "\n" . $e->getTraceAsString());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>