<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}

try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key only
    validateApiKey();
    
    // Pagination parameters
    $page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
    $limit = isset($_GET['limit']) ? max(1, min(100, intval($_GET['limit']))) : 10;
    $offset = ($page - 1) * $limit;
    
    // Count total records
    $countQuery = "SELECT COUNT(*) as total FROM trx_news";
    $countStmt = $pdo->prepare($countQuery);
    $countStmt->execute();
    $totalData = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Query with pagination
    $query = "
        SELECT id, trxdate, title, briefdesc, bannerfile
        FROM trx_news
        ORDER BY trxdate DESC
        LIMIT :limit OFFSET :offset
    ";
    
    // Execute query
    $stmt = $pdo->prepare($query);
    $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $news = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Process results to add image URLs
    $baseUrl = getenv('BASE_URL') ?: 'https://api.ezlink.dev/matrixweb/assets';
    foreach ($news as &$item) {
        if (!empty($item['bannerfile'])) {
            $item['imageurl'] = $baseUrl . '/news/' . $item['bannerfile'];
        } else {
            $item['imageurl'] = $baseUrl . '/news/default.jpg';
        }
        
        // Format date
        if (!empty($item['trxdate'])) {
            $item['formatted_date'] = date('d M Y', strtotime($item['trxdate']));
        }
        
        // Remove bannerfile from response
        unset($item['bannerfile']);
    }
    
    // Format the response
    $response = [
        "success" => true,
        "data" => [
            "news" => $news,
            "total_data" => intval($totalData),
            "current_page" => $page,
            "limit" => $limit
        ]
    ];
    
    // Send successful response
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("News API error: " . $e->getMessage() . "\n" . $e->getTraceAsString());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}

/**
 * Send JSON response with appropriate headers
 *
 * @param array $data The data to encode as JSON
 * @param int $statusCode HTTP status code
 * @return void
 */

?>