<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}

try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
      
    // Validate required parameter
    if (!isset($_GET['slug']) || empty($_GET['slug'])) {
        throw new Exception("Landing page slug is required", 400);
    }
    
    // Get and sanitize slug parameter
    $slug = trim($_GET['slug']);
    
    // Query landing page data
    $landingPageStmt = $pdo->prepare("
        SELECT id, slug, wpurl, metatitle, metadesc, metakeyword
        FROM mst_landingpage 
        WHERE slug = :slug 
    ");
    $landingPageStmt->bindParam(':slug', $slug, PDO::PARAM_STR);
    $landingPageStmt->execute();
    
    // Check if landing page exists
    if ($landingPageStmt->rowCount() === 0) {
        throw new Exception("Landing page not found", 404);
    }
    
    $landingPageData = $landingPageStmt->fetch(PDO::FETCH_ASSOC);
    
    // Format the response
    $response = [
        "success" => true,
        "message" => "Landing page retrieved successfully",
        "data" => [
            "id" => intval($landingPageData['id']),
            "slug" => $landingPageData['slug'],
            "wpurl" => $landingPageData['wpurl'],
            "metatitle" => $landingPageData['metatitle'],
            "metadesc" => $landingPageData['metadesc'],
            "metakeyword" => $landingPageData['metakeyword']
        ]
    ];
    
    // Send successful response
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("View Landing Page API error: " . $e->getMessage());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>