<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}

try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
    // Query to get katalog data
    $query = "
        SELECT id, docname, note, bannerfile, filedoc
        FROM mst_katalog
        ORDER BY id DESC
    ";
    
    // Execute query
    $stmt = $pdo->prepare($query);
    $stmt->execute();
    $katalogList = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get base URL from config for files
    $baseUrl = getenv('BASE_URL') ?: 'https://api.ezlink.dev/matrixweb/assets';
    
    // Format the response
    $katalog = array();
    foreach ($katalogList as $item) {
        // Set image URL for banner
        $imageUrl = null;
        if (!empty($item['bannerfile'])) {
            $imageUrl = $baseUrl . '/katalog/' . $item['bannerfile'];
        }
        
        // Set PDF file URL
        $pdfUrl = null;
        if (!empty($item['filedoc'])) {
            $pdfUrl = $baseUrl . '/katalog/' . $item['filedoc'];
        }
        
        $katalogObj = array(
            'id' => intval($item['id']),
            'title' => $item['docname'] ?? '',
            'briefdesc' => $item['note'] ?? '',
            'imageurl' => $imageUrl,
            'pdffile' => $pdfUrl
        );
        
        array_push($katalog, $katalogObj);
    }
    
    // Format the response
    $response = [
        "success" => true,
        "data" => [
            "katalog" => $katalog
        ]
    ];
    
    // Send successful response
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("Katalog API error: " . $e->getMessage() . "\n" . $e->getTraceAsString());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>