<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}

try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
        
    // News
    $query = "
        SELECT id, trxdate, title, briefdesc, bannerfile
        FROM trx_news
        ORDER BY trxdate DESC
        LIMIT 5
    ";

    // Execute query
    $stmt = $pdo->prepare($query);
    $stmt->execute();
    $news = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Process results to add image URLs
    $baseUrl = getenv('BASE_URL') ?: 'https://api.ezlink.dev/matrixweb/assets';
    foreach ($news as &$item) {
        if (!empty($item['bannerfile'])) {
            $item['imageurl'] = $baseUrl . '/news/' . $item['bannerfile'];
        } else {
            $item['imageurl'] = $baseUrl . '/news/default.jpg';
        }
        
        // Format date
        if (!empty($item['trxdate'])) {
            $item['formatted_date'] = date('d M Y', strtotime($item['trxdate']));
        }
        
        // Remove bannerfile from response
        unset($item['bannerfile']);
    }

    // select slider from mst_slider where in range startdate and enddate
    $currentDate = date('Y-m-d');
    $sliderQuery = "
        SELECT id, slidername, imagefile, startdate, enddate, targeturl
        FROM mst_slider
        WHERE (startdate <= :currentDate OR startdate IS NULL) 
        AND (enddate >= :currentDate OR enddate IS NULL)
        ORDER BY id
    ";
    
    $sliderStmt = $pdo->prepare($sliderQuery);
    $sliderStmt->bindParam(':currentDate', $currentDate, PDO::PARAM_STR);
    $sliderStmt->execute();
    $sliders = $sliderStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Process slider results to add image URLs
    $slider = array();
    foreach ($sliders as $sliderItem) {
        $sliderObj = array(
            'id' => intval($sliderItem['id']),
            'slidername' => $sliderItem['slidername'],
            'targeturl' => $sliderItem['targeturl'] // Include targeturl in response
        );
        
        if (!empty($sliderItem['imagefile'])) {
            $sliderObj['imageurl'] = $baseUrl . '/slider/' . $sliderItem['imagefile'];
        } else {
            $sliderObj['imageurl'] = $baseUrl . '/slider/default.jpg';
        }
        
        // Add to slider array
        $slider[] = $sliderObj;
    }

    // Get productnex from database (mst_paketnex and mst_paketnexdet)
    $productnex = array();
    
    // Get active paket nex within date range
    $paketQuery = "
        SELECT p.id, p.namapaket, p.bannerfile, nc.category as categoryname
        FROM mst_paketnex p
        LEFT JOIN mst_nexcategory nc ON p.categoryid = nc.id
        WHERE (p.startdate <= :currentDate OR p.startdate IS NULL) 
        AND (p.enddate >= :currentDate OR p.enddate IS NULL)
        ORDER BY p.namapaket ASC
    ";
    
    $paketStmt = $pdo->prepare($paketQuery);
    $paketStmt->bindParam(':currentDate', $currentDate, PDO::PARAM_STR);
    $paketStmt->execute();
    $paketList = $paketStmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach ($paketList as $paket) {
        // Get paket details (pricing options)
        $detailQuery = "
            SELECT id, kodepaket, masaberlaku, price
            FROM mst_paketnexdet
            WHERE paketid = :paketid
            ORDER BY price ASC
        ";
        
        $detailStmt = $pdo->prepare($detailQuery);
        $detailStmt->bindParam(':paketid', $paket['id'], PDO::PARAM_INT);
        $detailStmt->execute();
        $paketDetails = $detailStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Find minimum price for display
        $minPrice = 0;
        if (!empty($paketDetails)) {
            $minPrice = min(array_column($paketDetails, 'price'));
        }
        
        // Set banner image URL
        $imageUrl = $baseUrl . '/paketnex/default.jpg'; // Default image
        if (!empty($paket['bannerfile'])) {
            $imageUrl = $baseUrl . '/paketnex/' . $paket['bannerfile'];
        }
        
        // Format paket details
        $formattedDetails = array();
        foreach ($paketDetails as $detail) {
            $detailObj = array(
                'kodepaket' => $detail['kodepaket'],
                'masaberlaku' => $detail['masaberlaku'],
                'price' => "Rp. " . number_format($detail['price'], 0)
            );
            $formattedDetails[] = $detailObj;
        }
        
        // Create paket object
        $paketObj = array(
            'id' => intval($paket['id']),
            'paketname' => $paket['namapaket'],
            'paketdesc' => $paket['categoryname'] ?? 'Paket berlangganan',
            'price' => $minPrice > 0 ? "Mulai Rp. " . number_format($minPrice, 0) : "Hubungi Kami",
            'imageurl' => $imageUrl,
            'detail' => $formattedDetails
        );
        
        $productnex[] = $paketObj;
    }

    // Get home products from database (ishome = 1)
    $product = array();
    $productQuery = "
        SELECT p.id, p.itemname, p.briefdesc, p.price, p.pricepromo, p.slug
        FROM mst_product p
        WHERE p.ishome = 1 AND p.status = 1
        ORDER BY p.itemname
        LIMIT 6
    ";
    
    $productStmt = $pdo->prepare($productQuery);
    $productStmt->execute();
    $homeProducts = $productStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Process product results
    foreach ($homeProducts as $prod) {
        // Get product photo
        $photoQuery = "
            SELECT photo 
            FROM mst_productphoto 
            WHERE productid = :productid 
            ORDER BY seqno ASC 
            LIMIT 1
        ";
        $photoStmt = $pdo->prepare($photoQuery);
        $photoStmt->bindParam(':productid', $prod['id'], PDO::PARAM_INT);
        $photoStmt->execute();
        $photo = $photoStmt->fetch(PDO::FETCH_ASSOC);
        
        // Set image URL
        if ($photo && !empty($photo['photo'])) {
            $imageUrl = $baseUrl . '/product/' . $photo['photo'];
        } else {
            // Default image if product has no photos
            $imageUrl = $baseUrl . '/home/default-product.jpg';
        }
        
        // Get average rating for product
        $ratingQuery = "
            SELECT AVG(rating) as avg_rating 
            FROM mst_productreview 
            WHERE productid = :productid
        ";
        $ratingStmt = $pdo->prepare($ratingQuery);
        $ratingStmt->bindParam(':productid', $prod['id'], PDO::PARAM_INT);
        $ratingStmt->execute();
        $ratingResult = $ratingStmt->fetch(PDO::FETCH_ASSOC);
        $avgRating = $ratingResult['avg_rating'] ? round($ratingResult['avg_rating'], 1) : 0;
        
        // Format price and normal price
        if (isset($prod['pricepromo']) && $prod['pricepromo'] > 0 && $prod['pricepromo'] < $prod['price']) {
            $price = "Rp. " . number_format($prod['pricepromo'], 0);
            $pricenormal = "Rp. " . number_format($prod['price'], 0);
        } else {
            $price = "Rp. " . number_format($prod['price'], 0);
            $pricenormal = "";
        }
        
        // Create product object
        $productObj = array(
            'productid' => intval($prod['id']),
            'productname' => $prod['itemname'],
            'briefdesc' => $prod['briefdesc'],
            'price' => $price,
            'pricenormal' => $pricenormal,
            'imageurl' => $imageUrl,
            'slug' => $prod['slug'], // Include slug in response
            'rating' => $avgRating
        );
        
        // Add to product array
        $product[] = $productObj;
    }

    // Get featured products from database (isfeature = 1)
    $highlight = array();
    $highlightQuery = "
        SELECT p.id, p.itemname, p.briefdesc, p.price, p.pricepromo, p.slug
        FROM mst_product p
        WHERE p.isfeature = 1 AND p.status = 1
        ORDER BY p.id limit 5
    ";
    
    $highlightStmt = $pdo->prepare($highlightQuery);
    $highlightStmt->execute();
    $featuredProducts = $highlightStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Process featured products
    $mainHighlightSet = false;
    foreach ($featuredProducts as $featured) {
        // Get product photo
        $photoQuery = "
            SELECT photo 
            FROM mst_productphoto 
            WHERE productid = :productid 
            ORDER BY seqno ASC 
            LIMIT 1
        ";
        $photoStmt = $pdo->prepare($photoQuery);
        $photoStmt->bindParam(':productid', $featured['id'], PDO::PARAM_INT);
        $photoStmt->execute();
        $photo = $photoStmt->fetch(PDO::FETCH_ASSOC);
        
        // Set image URL
        if ($photo && !empty($photo['photo'])) {
            $imageUrl = $baseUrl . '/product/' . $photo['photo'];
        } else {
            // Default image if product has no photos
            $imageUrl = $baseUrl . '/home/default-product.jpg';
        }
        

        
        // Format price and normal price
        if (isset($featured['pricepromo']) && $featured['pricepromo'] > 0 && $featured['pricepromo'] < $featured['price']) {
            $price = "Rp. " . number_format($featured['pricepromo'], 0);
            $pricenormal = "Rp. " . number_format($featured['price'], 0);
        } else {
            $price = "Rp. " . number_format($featured['price'], 0);
            $pricenormal = "";
        }
        
        // Create highlight object
        $highlightObj = array(
            'id' => intval($featured['id']),
            'title' => $featured['itemname'],
            'briefdesc' => $featured['briefdesc'],
            'price' => $price,
            'pricenormal' => $pricenormal,
            'imageurl' => $imageUrl,
            'slug' => $featured['slug'] // Include slug in response
        );
        
        // First product is main highlight
        if (!$mainHighlightSet) {
            $highlightObj['main'] = 1;
            $mainHighlightSet = true;
        } else {
            $highlightObj['main'] = 0;
        }
        
        // Add to highlight array
        $highlight[] = $highlightObj;
    }

    // Format the response
    $response = [
        "success" => true,
        "data" => [
            "slider" => $slider,
            "productnex" => $productnex,
            "product" => $product,
            "news" => $news,
            "highlight" => $highlight,
        ]
    ];
    
    // Send successful response
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("Home API error: " . $e->getMessage() . "\n" . $e->getTraceAsString());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>