<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}



try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
    // Validate required parameter
    if (!isset($_GET['id']) || empty($_GET['id'])) {
        throw new Exception("Highlight ID is required", 400);
    }
    
    // Get highlight ID
    $highlightId = intval($_GET['id']);
    
    // Query highlight data
    $highlightStmt = $pdo->prepare("
        SELECT id, trxdate, title, briefdesc, content, bannerfile
        FROM trx_highlight 
        WHERE id = :id 
    ");
    $highlightStmt->bindParam(':id', $highlightId, PDO::PARAM_INT);
    $highlightStmt->execute();
    
    // Check if highlight exists
    if ($highlightStmt->rowCount() === 0) {
        throw new Exception("Highlight not found", 404);
    }
    
    $highlightData = $highlightStmt->fetch(PDO::FETCH_ASSOC);
    
    // Format date
    if (!empty($highlightData['trxdate'])) {
        $highlightData['trxdate'] = date('Y-m-d', strtotime($highlightData['trxdate']));
    }
    
    // Get base URL from config for banner images
    $baseUrl = getenv('BASE_URL') ?: 'https://api.ezlink.dev/matrixweb/assets';
    
    // Set full banner URL if banner exists
    if (!empty($highlightData['bannerfile'])) {
        $highlightData['banner_url'] = $baseUrl . '/highlight/' . $highlightData['bannerfile'];
    } else {
        $highlightData['banner_url'] = null;
    }
    
    // Remove bannerfile from response
    unset($highlightData['bannerfile']);
    
    // Format the response
    $response = [
        "success" => true,
        "message" => "Highlight retrieved successfully",
        "data" => $highlightData
    ];
    
    // Send successful response
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("View Highlight API error: " . $e->getMessage());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>