<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}


try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
    // Query to get highlight data ordered by trxdate desc
    $query = "
        SELECT id, trxdate, title, briefdesc, bannerfile
        FROM trx_highlight
        ORDER BY trxdate DESC
    ";
    
    // Execute query
    $stmt = $pdo->prepare($query);
    $stmt->execute();
    $highlightList = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get base URL from config for banner images
    $baseUrl = getenv('BASE_URL') ?: 'https://api.ezlink.dev/matrixweb/assets';
    $baseUrl = rtrim($baseUrl, '/') . '/highlight/';
    
    // Format the response and prepare banner URLs
    foreach ($highlightList as &$highlight) {
        // Format date
        if (!empty($highlight['trxdate'])) {
            $highlight['trxdate'] = date('Y-m-d', strtotime($highlight['trxdate']));
        }
        
        // Set full banner URL if banner exists
        if (!empty($highlight['bannerfile'])) {
            $highlight['banner_url'] = $baseUrl . $highlight['bannerfile'];
        } else {
            $highlight['banner_url'] = null;
        }
        
        // Remove bannerfile from response (keep id)
        unset($highlight['bannerfile']);
    }
    
    // Format the response
    $response = [
        "success" => true,
        "data" => $highlightList
    ];
    
    // Send successful response
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("Highlight API error: " . $e->getMessage() . "\n" . $e->getTraceAsString());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>