<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}



try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
    // Get and parse JSON input
    $jsonData = getJsonRequest();
    
    // Validate required fields
    $missingParams = [];
    
    if (empty($jsonData['username'])) {
        $missingParams['username'] = 'required';
    }
    
    if (empty($jsonData['email'])) {
        $missingParams['email'] = 'required';
    }
    
    if (empty($jsonData['hpno'])) {
        $missingParams['hpno'] = 'required';
    }
    
    if (empty($jsonData['messages'])) {
        $missingParams['messages'] = 'required';
    }
    
    // Check if there are missing parameters
    if (!empty($missingParams)) {
        $response = [
            "success" => false,
            "message" => "Missing required parameters",
            "data" => [
                "missing_parameters" => $missingParams
            ]
        ];
        sendJsonResponse($response, 400);
        exit;
    }
    
    // Sanitize input
    $username = htmlspecialchars(trim($jsonData['username']), ENT_QUOTES, 'UTF-8');
    $email = trim($jsonData['email']);
    $hpno = trim($jsonData['hpno']);
    $messages = htmlspecialchars(trim($jsonData['messages']), ENT_QUOTES, 'UTF-8');
    
    // Validate username length
    if (strlen($username) > 100) {
        throw new Exception("Username must be less than 100 characters", 400);
    }
    
    // Validate email format and length
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        throw new Exception("Invalid email format", 400);
    }
    
    if (strlen($email) > 100) {
        throw new Exception("Email must be less than 100 characters", 400);
    }
    
    // Validate phone number length
    if (strlen($hpno) > 25) {
        throw new Exception("Phone number must be less than 25 characters", 400);
    }
    
    // Validate messages length
    if (strlen($messages) > 1000) {
        throw new Exception("Message must be less than 1000 characters", 400);
    }
    
    // Insert new message
    $insertStmt = $pdo->prepare("
        INSERT INTO trx_messages (username, email, hpno, messages) 
        VALUES (:username, :email, :hpno, :messages)
    ");
    
    $insertStmt->bindParam(':username', $username, PDO::PARAM_STR);
    $insertStmt->bindParam(':email', $email, PDO::PARAM_STR);
    $insertStmt->bindParam(':hpno', $hpno, PDO::PARAM_STR);
    $insertStmt->bindParam(':messages', $messages, PDO::PARAM_STR);
    $insertStmt->execute();
    
    // Get the new message ID
    $messageId = $pdo->lastInsertId();
    
    // Success response
    $response = [
        "success" => true,
        "message" => "Pesan Anda sudah kami terima.",
        "data" => [
            "id" => intval($messageId),
            "username" => $username,
            "email" => $email,
            "hpno" => $hpno,
            "messages" => $messages
        ]
    ];
    
    // Send successful response
    sendJsonResponse($response, 201); // Created
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("Add Message API error: " . $e->getMessage());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>