<?php

// Strict error reporting in development, disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include helpers
require_once __DIR__ . "/../connection.php";
require_once __DIR__ . "/../auth.php";

// Set CORS headers
setCorsHeaders();

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405); // Method Not Allowed
    exit(json_encode(['success' => false, 'message' => 'Method Not Allowed']));
}

try {
    // Get PDO connection
    $pdo = connectDB();
    
    // Validate API key
    validateApiKey();
    
    // Validate authentication token
    $authenticatedUser = validateAuthToken($pdo);
    
    // Check if pagination is required
    $isPaging = isset($_GET['ispaging']) ? intval($_GET['ispaging']) : 1; // Default to using pagination
    
    // Base query with JOIN to get golongan name
    $baseWhere = "WHERE 1=1";
    
    // Handle golonganid filter
    if (isset($_GET['golonganid']) && !empty($_GET['golonganid'])) {
        $golonganId = intval($_GET['golonganid']);
        $baseWhere .= " AND c.golonganid = $golonganId";
    } else if ($isPaging == 0) {
        // If isPaging is 0, golonganid must exist
        throw new Exception("Golongan ID is required when isPaging is 0", 400);
    }
    
    // Build the query with join to get golongan name
    $query = "
        SELECT c.id, c.category, c.golonganid, g.golongan,
               (SELECT COUNT(*) FROM mst_subcategory s WHERE s.categoryid = c.id) as subcategory_count
        FROM mst_category c
        LEFT JOIN mst_golongan g ON c.golonganid = g.id
        $baseWhere
    ";
    
    $countQuery = "
        SELECT COUNT(*) as total
        FROM mst_category c
        $baseWhere
    ";
    
    // Search parameter
    $search = isset($_GET['search']) ? trim($_GET['search']) : '';
    
    // Add search condition if provided and if isPaging = 1
    if ($isPaging == 1 && !empty($search)) {
        $searchTerm = "%$search%";
        $query .= " AND c.category LIKE '$searchTerm'";
        $countQuery .= " AND c.category LIKE '$searchTerm'";
    }
    
    // Sorting parameters
    $allowedSortFields = ['id', 'category', 'golongan', 'subcategory_count'];
    $sortBy = isset($_GET['sort_by']) && in_array($_GET['sort_by'], $allowedSortFields) ? $_GET['sort_by'] : 'category';
    $sortKey = isset($_GET['sort_key']) && strtoupper($_GET['sort_key']) === 'DESC' ? 'DESC' : 'ASC';
    
    // Add sorting
    $query .= " ORDER BY $sortBy $sortKey";
    
    // If pagination is required
    if ($isPaging == 1) {
        // Pagination parameters
        $page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
        $pageSize = isset($_GET['page_size']) ? max(1, min(100, intval($_GET['page_size']))) : 10;
        $offset = ($page - 1) * $pageSize;
        
        // Execute count query
        $countStmt = $pdo->prepare($countQuery);
        $countStmt->execute();
        $totalCount = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
        
        // Add pagination to query
        $query .= " LIMIT $pageSize OFFSET $offset";
        
        // Execute main query
        $stmt = $pdo->prepare($query);
        $stmt->execute();
        $categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Calculate pagination details
        $totalPages = ceil($totalCount / $pageSize);
        
        // Format the response with pagination
        $response = [
            "success" => true,
            "data" => [
                "list" => $categories,
                "pagination" => [
                    "total" => $totalCount,
                    "page_size" => $pageSize,
                    "current_page" => $page,
                    "last_page" => $totalPages,
                    "from" => ($totalCount > 0) ? $offset + 1 : 0,
                    "to" => min($offset + $pageSize, $totalCount)
                ],
                "sort" => [
                    "sort_by" => $sortBy,
                    "sort_key" => $sortKey
                ]
            ]
        ];
    } else {
        // No pagination, return all records
        $stmt = $pdo->prepare($query);
        $stmt->execute();
        $categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Format the response without pagination
        $response = [
            "success" => true,
            "data" => $categories
        ];
    }
    
    // Send successful response
    sendJsonResponse($response, 200);
    
} catch (Exception $e) {
    $errorCode = $e->getCode();
    $errorCode = is_int($errorCode) ? $errorCode : 500;
    
    // Prepare error response
    $response = [
        "success" => false,
        "message" => $e->getMessage(),
        "data" => null
    ];
    
    // Hide internal error details from users
    if ($errorCode >= 500) {
        error_log("List Categories API error: " . $e->getMessage() . "\n" . $e->getTraceAsString());
        $response["message"] = "An internal error occurred";
    }
    
    // Send error response with appropriate status code
    $statusCode = ($errorCode >= 400 && $errorCode < 600) ? $errorCode : 500;
    sendJsonResponse($response, $statusCode);
}
?>