<?php

require_once __DIR__ . '/vendor/autoload.php';
$dotenv = Dotenv\Dotenv::createImmutable(__DIR__);
$dotenv->load();

/**
 * Authentication helper functions for API endpoints
 */

/**
 * Validates a user's authentication token
 * 
 * @param PDO $pdo Database connection
 * @param string|null $authHeader The Authorization header value
 * @return array User information if token is valid
 * @throws Exception If token is invalid, missing, or expired
 */
function validateAuthToken(PDO $pdo, ?string $authHeader = null)
{
    // Get token from header if not directly provided
    if ($authHeader === null) {
        $authHeader = $_SERVER['HTTP_AUTHORIZATION'] ?? null;
    }
    
    // Check if token exists
    if (empty($authHeader)) {
        throw new Exception("Authentication required", 401);
    }
    
    // Extract token (remove 'Bearer ' prefix if present)
    $token = $authHeader;
    if (strpos($authHeader, 'Bearer ') === 0) {
        $token = substr($authHeader, 7);
    }
    
    // Validate token from database
    $stmt = $pdo->prepare("SELECT id, userid, username FROM mst_user 
                          WHERE sessionid = :token 
                          AND token_expiry > NOW()");
    $stmt->bindParam(':token', $token, PDO::PARAM_STR);
    $stmt->execute();
    
    if ($stmt->rowCount() === 0) {
        throw new Exception("Invalid or expired token", 401);
    }
    
    // Return authenticated user info
    return $stmt->fetch();
}

/**
 * Validates API key
 * 
 * @param string|null $apiKey The API key to validate
 * @return bool True if valid
 * @throws Exception If API key is invalid or missing
 */
function validateApiKey(?string $apiKey = null)
{
    // Get API key from header if not directly provided
    if ($apiKey === null) {
        $apiKey = $_SERVER['HTTP_X_APIKEY'] ?? null;
    }
    
    // Get valid API key from environment
    $validApiKey = $_ENV['API_KEY'] ?? getenv('API_KEY') ?? null;
    
    // Validate
    if (empty($apiKey) || empty($validApiKey) || !hash_equals($validApiKey, $apiKey)) {
        throw new Exception("Invalid API key", 401);
    }
    
    return true;
}

/**
 * Sets CORS headers for API responses
 */
function setCorsHeaders()
{
    $allowedOrigin = $_ENV['ALLOWED_ORIGIN'] ?? 'https://yourdomain.com';
    header("Access-Control-Allow-Origin: *");
    header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
    header("Access-Control-Allow-Headers: Content-Type, X-APIKey, Authorization");
    header("Access-Control-Max-Age: 3600");
    
    // Handle preflight OPTIONS request
    if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
        http_response_code(204); // No Content
        exit();
    }
}

/**
 * Validates and processes JSON request
 * 
 * @return array Decoded JSON data
 * @throws Exception If JSON is invalid
 */
function getJsonRequest()
{
    // Check content type
    if (!isset($_SERVER['CONTENT_TYPE']) || strpos($_SERVER['CONTENT_TYPE'], 'application/json') === false) {
        throw new Exception("Unsupported Media Type", 415);
    }
    
    // Get and parse JSON
    $jsonInput = file_get_contents('php://input');
    $jsonData = json_decode($jsonInput, true);
    
    if (json_last_error() !== JSON_ERROR_NONE || empty($jsonData)) {
        throw new Exception("Invalid request format: " . json_last_error_msg(), 400);
    }
    
    return $jsonData;
}

/**
 * Sends JSON response with appropriate headers
 * 
 * @param array $data Response data
 * @param int $statusCode HTTP status code
 */
function sendJsonResponse(array $data, int $statusCode = 200)
{
    http_response_code($statusCode);
    header('Content-Type: application/json; charset=UTF-8');
    echo json_encode($data);
    exit;
}
?>